// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright The XCSoar Project

#pragma once

#include "GlideComputerBlackboard.hpp"
#include "time/PeriodClock.hpp"
#include "time/DeltaTime.hpp"
#include "GlideComputerAirData.hpp"
#include "StatsComputer.hpp"
#include "TaskComputer.hpp"
#include "LogComputer.hpp"
#include "WarningComputer.hpp"
#include "CuComputer.hpp"
#include "Engine/Contest/Solvers/Retrospective.hpp"
#include "ConditionMonitor/ConditionMonitors.hpp"
#include "ConditionMonitor/MoreConditionMonitors.hpp"

class Waypoints;
class ProtectedTaskManager;
class GlideComputerTaskEvents;
class RasterTerrain;

// TODO: replace copy constructors so copies of these structures
// do not replicate the large items or items that should be singletons
// OR: just make them static?

class GlideComputer : public GlideComputerBlackboard
{
  GlideComputerAirData air_data_computer;
  WarningComputer warning_computer;
  TaskComputer task_computer;
  StatsComputer stats_computer;
  LogComputer log_computer;
  CuComputer cu_computer;

  ConditionMonitors condition_monitors;
  MoreConditionMonitors idle_condition_monitors;

  const Waypoints &waypoints;

  Retrospective retrospective;
  int team_code_ref_id;
  bool team_code_ref_found;
  GeoPoint team_code_ref_location;

  PeriodClock idle_clock;

  /**
   * This object is used to check whether to update
   * DerivedInfo::trace_history.
   */
  DeltaTime trace_history_time;

public:
  GlideComputer(const ComputerSettings &_settings,
                const Waypoints &_way_points,
                Airspaces &_airspace_database,
                ProtectedTaskManager& task,
                GlideComputerTaskEvents& events);

  void SetTerrain(RasterTerrain *_terrain);

  void SetLogger(Logger *logger) {
    log_computer.SetLogger(logger);
  }

  /**
   * Resets the GlideComputer data
   * @param full Reset all data?
   */
  void ResetFlight(const bool full=true);

  /**
   * Initializes the GlideComputer
   */
  void Initialise();

  void Expire() {
    SetCalculated().Expire(Basic().clock);
  }

  /**
   * Is called by the CalculationThread and processes the received GPS
   * data in Basic().
   *
   * @param force forces calculation even if there was no new GPS fix
   */
  bool ProcessGPS(bool force=false); // returns true if idle needs processing

  /**
   * Process slow calculations. Called by the CalculationThread.
   */
  void ProcessIdle(bool exhaustive=false);

  void ProcessExhaustive() {
    ProcessIdle(true);
  }

  void OnStartTask();
  void OnFinishTask();
  void OnTransitionEnter();

  const WindStore &GetWindStore() const {
    return air_data_computer.GetWindStore();
  }

  const CuSonde &GetCuSonde() const {
    return cu_computer.GetCuSonde();
  }

  ProtectedAirspaceWarningManager &GetAirspaceWarnings() {
    return warning_computer.GetManager();
  }

  const ProtectedAirspaceWarningManager &GetAirspaceWarnings() const {
    return warning_computer.GetManager();
  }

  const TraceComputer &GetTraceComputer() const {
    return task_computer.GetTraceComputer();
  }

  const ProtectedTaskManager &GetProtectedTaskManager() const {
    return task_computer.GetProtectedTaskManager();
  }

  const ProtectedRoutePlanner &GetProtectedRoutePlanner() const {
    return task_computer.GetProtectedRoutePlanner();
  }

  void ClearAirspaces() {
    task_computer.ClearAirspaces();
  }

  const FlightStatistics &GetFlightStats() const {
    return stats_computer.GetFlightStats();
  }

  const Retrospective &GetRetrospective() const {
    return retrospective;
  }

  void SetContestIncremental(bool incremental) {
    task_computer.SetContestIncremental(incremental);
  }

protected:
  void OnTakeoff();
  void OnLanding();

  /**
   * Detects takeoff and landing events
   */
  void TakeoffLanding(bool last_flying);

private:

  /**
   * Fill the cache variable TeamCodeRefLocation.
   *
   * @return true if the location was found, false if the
   * TeamCodeRefLocation variable is undefined
   */
  [[gnu::pure]]
  bool DetermineTeamCodeRefLocation();

  void CalculateTeammateBearingRange();

  /**
   * Calculates the own TeamCode and saves it to Calculated
   */
  void CalculateOwnTeamCode();

  void CalculateWorkingBand();
  void CalculateVarioScale();
};
