// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright The XCSoar Project

#include "Input.hpp"
#include "MergeMouse.hpp"
#include "ui/event/shared/Event.hpp"
#include "ui/event/Queue.hpp"
#include "io/FileDescriptor.hxx"
#include "Asset.hpp"
#include "Translate.hpp"

#include <algorithm>

#include <termios.h>
#include <sys/ioctl.h>
#include <errno.h>

template<typename T>
static constexpr unsigned
BitSize() noexcept
{
  return 8 * sizeof(T);
}

template<typename T>
static constexpr size_t
BitsToInts(unsigned n_bits) noexcept
{
  return (n_bits + BitSize<T>() - 1) / BitSize<T>();
}

template<typename T>
static constexpr bool
CheckBit(const T bits[], unsigned i) noexcept
{
  return bits[i / BitSize<T>()] & (T(1) << (i % BitSize<T>()));
}

namespace UI {

LinuxInputDevice::LinuxInputDevice(EventQueue &_queue,
                                   MergeMouse &_merge) noexcept
  :queue(_queue), merge(_merge),
   edit_position(0, 0), public_position(0, 0),
   event(queue.GetEventLoop(), BIND_THIS_METHOD(OnSocketReady))
{
}

/**
 * Check if the EVDEV supports EV_ABS or EV_REL..
 */
[[gnu::pure]]
static bool
IsPointerDevice(int fd) noexcept
{
  assert(fd >= 0);

  unsigned long features[BitsToInts<unsigned long>(std::max(EV_ABS, EV_REL))];
  if (ioctl(fd, EVIOCGBIT(0, sizeof(features)), features) < 0)
    return false;

  return CheckBit(features, EV_ABS) || CheckBit(features, EV_REL);
}

bool
LinuxInputDevice::Open(const char *path) noexcept
{
  FileDescriptor _fd;
  if (!_fd.OpenReadOnly(path))
    return false;

  _fd.SetNonBlocking();
  event.Open(_fd);
  event.ScheduleRead();

  min_x = max_x = min_y = max_y = 0;

  is_pointer = IsPointerDevice(event.GetFileDescriptor().Get());
  if (is_pointer) {
    merge.AddPointer();

    if (!IsKobo()) {
      /* obtain touch screen information */
      /* no need to do that on the Kobo, because we know its touch
         screen is well-calibrated */

      const int fd = event.GetFileDescriptor().Get();

      input_absinfo abs;
      if (ioctl(fd, EVIOCGABS(ABS_X), &abs) == 0) {
        min_x = abs.minimum;
        max_x = abs.maximum;
      }

      if (ioctl(fd, EVIOCGABS(ABS_Y), &abs) == 0) {
        min_y = abs.minimum;
        max_y = abs.maximum;
      }
    }
  }

  rel_x = rel_y = rel_wheel = 0;
  down = false;
  moving = pressing = releasing = false;
  return true;
}

void
LinuxInputDevice::Close() noexcept
{
  if (!IsOpen())
    return;

  if (is_pointer)
    merge.RemovePointer();

  event.Close();
}

inline void
LinuxInputDevice::Read() noexcept
{
  FileDescriptor fd = event.GetFileDescriptor();

  struct input_event buffer[64];
  const auto nbytes = fd.Read(std::as_writable_bytes(std::span{buffer}));
  if (nbytes < 0) {
    /* device has failed or was unplugged - bail out */
    if (errno != EAGAIN && errno != EINTR)
      Close();
    return;
  }

  unsigned n = size_t(nbytes) / sizeof(buffer[0]);

  for (unsigned i = 0; i < n; ++i) {
    const struct input_event &e = buffer[i];

    switch (e.type) {
    case EV_SYN:
      if (e.code == SYN_REPORT) {
        /* commit the finger movement */

        const bool pressed = pressing;
        const bool released = releasing;
        pressing = releasing = false;

        if (pressed)
          merge.SetDown(true);

        if (released)
          merge.SetDown(false);

        if (IsKobo() && released) {
          /* workaround: on the Kobo Touch N905B, releasing the touch
             screen reliably produces a finger position that is way
             off; in that case, ignore finger movement */
          moving = false;
          edit_position = public_position;
        }

        if (moving) {
          moving = false;
          public_position = edit_position;
          merge.MoveAbsolute(public_position.x, public_position.y,
                             min_x, max_x, min_y, max_y);
        } else if (rel_x != 0 || rel_y != 0) {
          merge.MoveRelative(PixelPoint(rel_x, rel_y));
          rel_x = rel_y = 0;
        }

        if (rel_wheel != 0) {
          merge.MoveWheel(rel_wheel);
          rel_wheel = 0;
        }

        queue.WakeUp();
      }

      break;

    case EV_KEY:
      if (e.code == BTN_TOUCH || e.code == BTN_MOUSE) {
        bool new_down = e.value;
        if (new_down != down) {
          down = new_down;
          if (new_down)
            pressing = true;
          else
            releasing = true;
        }
      } else {
        /* Discard all data on stdin to avoid that keyboard input data is read
         * on the executing shell. This fixes #3403. */
        tcflush(STDIN_FILENO, TCIFLUSH);

        const auto [translated_key_code, is_char] = TranslateKeyCode(e.code);
        Event ev(e.value ? Event::KEY_DOWN : Event::KEY_UP,
                 translated_key_code);
        ev.is_char = is_char;
        queue.Push(ev);
      }

      break;

    case EV_ABS:
      moving = true;

      switch (e.code) {
      case ABS_X:
        edit_position.x = e.value;
        break;

      case ABS_Y:
        edit_position.y = e.value;
        break;

      case ABS_MT_POSITION_X:
        edit_position.x = e.value;
        break;

      case ABS_MT_POSITION_Y:
        edit_position.y = e.value;
        break;
      }

      break;

    case EV_REL:
      switch (e.code) {
      case REL_X:
        rel_x += e.value;
        break;

      case REL_Y:
        rel_y += e.value;
        break;

      case REL_WHEEL:
        rel_wheel += e.value;
        break;
      }

      break;
    }
  }
}

void
LinuxInputDevice::OnSocketReady(unsigned) noexcept
{
  Read();
}

} // namespace UI
