// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright The XCSoar Project

#include "AbstractTaskFactory.hpp"
#include "Constraints.hpp"
#include "Task/Ordered/OrderedTask.hpp"
#include "Task/Ordered/Points/StartPoint.hpp"
#include "Task/Ordered/Points/AATPoint.hpp"
#include "Task/Ordered/Points/ASTPoint.hpp"
#include "Task/Ordered/Points/FinishPoint.hpp"
#include "Task/ObservationZones/LineSectorZone.hpp"
#include "Task/ObservationZones/KeyholeZone.hpp"
#include "Task/ObservationZones/CylinderZone.hpp"
#include "Task/ObservationZones/AnnularSectorZone.hpp"

#include <algorithm>

static constexpr double
GetOZSize(const ObservationZonePoint &oz) noexcept
{
  switch (oz.GetShape()) {
  case ObservationZone::Shape::SECTOR:
  case ObservationZone::Shape::SYMMETRIC_QUADRANT:
    return ((const SectorZone &)oz).GetRadius();

  case ObservationZone::Shape::LINE:
    return ((const LineSectorZone &)oz).GetLength();

  case ObservationZone::Shape::CYLINDER:
  case ObservationZone::Shape::MAT_CYLINDER:
    return ((const CylinderZone &)oz).GetRadius();

  case ObservationZone::Shape::ANNULAR_SECTOR:
    return ((const AnnularSectorZone &)oz).GetRadius();

  default:
    return -1;
  }
}

std::unique_ptr<OrderedTaskPoint>
AbstractTaskFactory::CreateMutatedPoint(const OrderedTaskPoint &tp,
                                        const TaskPointFactoryType newtype) const noexcept
{
  auto ozsize = GetOZSize(tp.GetObservationZone());
  return CreatePoint(newtype, tp.GetWaypointPtr(), ozsize, ozsize, ozsize);
}

TaskPointFactoryType
AbstractTaskFactory::GetMutatedPointType(const OrderedTaskPoint &tp) const noexcept
{
  const TaskPointFactoryType oldtype = GetType(tp);
  TaskPointFactoryType newtype = oldtype;

  switch (tp.GetType()) {
  case TaskPointType::START:
    if (!IsValidStartType(newtype))
      newtype = GetDefaultStartType();
    break;

  case TaskPointType::AST:
  case TaskPointType::AAT:
    if (!IsValidIntermediateType(newtype))
      newtype = GetDefaultIntermediateType();
    break;

  case TaskPointType::FINISH:
    if (!IsValidFinishType(newtype))
      newtype = GetDefaultFinishType();
    break;

  case TaskPointType::UNORDERED:
    break;
  }
  return newtype;
}

std::unique_ptr<StartPoint>
AbstractTaskFactory::CreateStart(std::unique_ptr<ObservationZonePoint> oz,
                                 WaypointPtr wp) const noexcept
{
  assert(wp);

  return std::make_unique<StartPoint>(std::move(oz), std::move(wp), behaviour,
                                      GetOrderedTaskSettings().start_constraints);
}

std::unique_ptr<FinishPoint>
AbstractTaskFactory::CreateFinish(std::unique_ptr<ObservationZonePoint> oz,
                                  WaypointPtr wp) const noexcept
{
  assert(wp);

  return std::make_unique<FinishPoint>(std::move(oz), std::move(wp),
                                       behaviour,
                                       GetOrderedTaskSettings().finish_constraints);
}

std::unique_ptr<AATPoint>
AbstractTaskFactory::CreateAATPoint(std::unique_ptr<ObservationZonePoint> oz,
                                    WaypointPtr wp) const noexcept
{
  assert(wp);

  return std::make_unique<AATPoint>(std::move(oz), std::move(wp), behaviour);
}

std::unique_ptr<ASTPoint>
AbstractTaskFactory::CreateASTPoint(std::unique_ptr<ObservationZonePoint> oz,
                                    WaypointPtr wp) const noexcept
{
  assert(wp);

  return std::make_unique<ASTPoint>(std::move(oz), std::move(wp), behaviour);
}

std::unique_ptr<StartPoint>
AbstractTaskFactory::CreateStart(WaypointPtr wp) const noexcept
{
  assert(wp);

  return CreateStart(GetDefaultStartType(), std::move(wp));
}

std::unique_ptr<IntermediateTaskPoint>
AbstractTaskFactory::CreateIntermediate(WaypointPtr wp) const noexcept
{
  assert(wp);

  if (constraints.homogeneous_tps && task.TaskSize() > 1) {
    TaskPointFactoryType type = GetType(task.GetPoint(1));
    if (IsValidIntermediateType(type))
      return CreateIntermediate(type, std::move(wp));
  }

  return CreateIntermediate(GetDefaultIntermediateType(), std::move(wp));
}

std::unique_ptr<FinishPoint>
AbstractTaskFactory::CreateFinish(WaypointPtr wp) const noexcept
{
  assert(wp);

  return CreateFinish(GetDefaultFinishType(), std::move(wp));
}

TaskPointFactoryType 
AbstractTaskFactory::GetType(const OrderedTaskPoint &point) const noexcept
{
  const ObservationZonePoint &oz = point.GetObservationZone();

  switch (point.GetType()) {
  case TaskPointType::START:
    switch (oz.GetShape()) {
    case ObservationZone::Shape::FAI_SECTOR:
    case ObservationZone::Shape::SYMMETRIC_QUADRANT:
      return TaskPointFactoryType::START_SECTOR;

    case ObservationZone::Shape::LINE:
      return TaskPointFactoryType::START_LINE;

    case ObservationZone::Shape::CYLINDER:
    case ObservationZone::Shape::MAT_CYLINDER:
    case ObservationZone::Shape::SECTOR:
    case ObservationZone::Shape::DAEC_KEYHOLE:
    case ObservationZone::Shape::CUSTOM_KEYHOLE:
    case ObservationZone::Shape::BGAFIXEDCOURSE:
    case ObservationZone::Shape::BGAENHANCEDOPTION:
    case ObservationZone::Shape::ANNULAR_SECTOR:
      return TaskPointFactoryType::START_CYLINDER;

    case ObservationZone::Shape::BGA_START:
      return TaskPointFactoryType::START_BGA;
    }
    break;

  case TaskPointType::AAT:
    switch (oz.GetShape()) {
    case ObservationZone::Shape::SECTOR:
    case ObservationZone::Shape::FAI_SECTOR:
    case ObservationZone::Shape::SYMMETRIC_QUADRANT:
    case ObservationZone::Shape::DAEC_KEYHOLE:
    case ObservationZone::Shape::BGAFIXEDCOURSE:
    case ObservationZone::Shape::BGAENHANCEDOPTION:
    case ObservationZone::Shape::BGA_START:
    case ObservationZone::Shape::LINE:
      return TaskPointFactoryType::AAT_SEGMENT;
    case ObservationZone::Shape::ANNULAR_SECTOR:
      return TaskPointFactoryType::AAT_ANNULAR_SECTOR;
    case ObservationZone::Shape::CYLINDER:
      return TaskPointFactoryType::AAT_CYLINDER;

    case ObservationZone::Shape::CUSTOM_KEYHOLE:
      return TaskPointFactoryType::AAT_KEYHOLE;

    case ObservationZone::Shape::MAT_CYLINDER:
      return TaskPointFactoryType::MAT_CYLINDER;
    }
    break;

  case TaskPointType::AST:
    switch (oz.GetShape()) {
    case ObservationZone::Shape::FAI_SECTOR:
      return TaskPointFactoryType::FAI_SECTOR;

    case ObservationZone::Shape::CUSTOM_KEYHOLE:
      return TaskPointFactoryType::CUSTOM_KEYHOLE;

    case ObservationZone::Shape::DAEC_KEYHOLE:
      return TaskPointFactoryType::DAEC_KEYHOLE;

    case ObservationZone::Shape::BGAFIXEDCOURSE:
      return TaskPointFactoryType::BGAFIXEDCOURSE_SECTOR;

    case ObservationZone::Shape::BGAENHANCEDOPTION:
      return TaskPointFactoryType::BGAENHANCEDOPTION_SECTOR;

    case ObservationZone::Shape::BGA_START:
    case ObservationZone::Shape::CYLINDER:
    case ObservationZone::Shape::MAT_CYLINDER:
    case ObservationZone::Shape::SECTOR:
    case ObservationZone::Shape::LINE:
    case ObservationZone::Shape::ANNULAR_SECTOR:
      return TaskPointFactoryType::AST_CYLINDER;

    case ObservationZone::Shape::SYMMETRIC_QUADRANT:
      return TaskPointFactoryType::SYMMETRIC_QUADRANT;
    }
    break;

  case TaskPointType::FINISH:
    switch (oz.GetShape()) {
    case ObservationZone::Shape::BGA_START:
    case ObservationZone::Shape::FAI_SECTOR:
    case ObservationZone::Shape::SYMMETRIC_QUADRANT:
      return TaskPointFactoryType::FINISH_SECTOR;

    case ObservationZone::Shape::LINE:
      return TaskPointFactoryType::FINISH_LINE;

    case ObservationZone::Shape::CYLINDER:
    case ObservationZone::Shape::MAT_CYLINDER:
    case ObservationZone::Shape::SECTOR:
    case ObservationZone::Shape::DAEC_KEYHOLE:
    case ObservationZone::Shape::CUSTOM_KEYHOLE:
    case ObservationZone::Shape::BGAFIXEDCOURSE:
    case ObservationZone::Shape::BGAENHANCEDOPTION:
    case ObservationZone::Shape::ANNULAR_SECTOR:
      return TaskPointFactoryType::FINISH_CYLINDER;
    }
    break;

  case TaskPointType::UNORDERED:
    /* obviously, when we check the type of an OrderedTaskPoint, we
       should never get type==UNORDERED */
    gcc_unreachable();
    break;
  }

  // fail, should never get here
  gcc_unreachable();
}

std::unique_ptr<OrderedTaskPoint>
AbstractTaskFactory::CreatePoint(const TaskPointFactoryType type,
                                 WaypointPtr wp) const noexcept
{
  return CreatePoint(type, std::move(wp), -1, -1, -1);
}

void
AbstractTaskFactory::GetPointDefaultSizes([[maybe_unused]] const TaskPointFactoryType type,
                                          double &start_radius,
                                          double &turnpoint_radius,
                                          double &finish_radius) const noexcept
{
  TaskBehaviour ob = this->behaviour;

  if (start_radius < 0)
    start_radius = ob.sector_defaults.start_radius;

  if (turnpoint_radius < 0)
    turnpoint_radius = ob.sector_defaults.turnpoint_radius;

  if (finish_radius < 0)
    finish_radius = ob.sector_defaults.finish_radius;
}

std::unique_ptr<OrderedTaskPoint>
AbstractTaskFactory::CreatePoint(const TaskPointFactoryType type,
                                 WaypointPtr wp,
                                 double start_radius,
                                 double turnpoint_radius,
                                 double finish_radius) const noexcept
{
  assert(wp);

  GetPointDefaultSizes(type, start_radius, turnpoint_radius, finish_radius);

  const GeoPoint location = wp->location;
  switch (type) {
  case TaskPointFactoryType::START_SECTOR:
    return CreateStart(SymmetricSectorZone::CreateFAISectorZone(location,
                                                                false),
                       std::move(wp));
  case TaskPointFactoryType::START_LINE:
    return CreateStart(std::make_unique<LineSectorZone>(location, start_radius),
                       std::move(wp));
  case TaskPointFactoryType::START_CYLINDER:
    return CreateStart(std::make_unique<CylinderZone>(location, start_radius),
                       std::move(wp));
  case TaskPointFactoryType::START_BGA:
    return CreateStart(KeyholeZone::CreateBGAStartSectorZone(location),
                       std::move(wp));
  case TaskPointFactoryType::FAI_SECTOR:
    return CreateASTPoint(SymmetricSectorZone::CreateFAISectorZone(location,
                                                                   true),
                          std::move(wp));
  case TaskPointFactoryType::SYMMETRIC_QUADRANT:
    return CreateASTPoint(std::make_unique<SymmetricSectorZone>(location,
                                                                turnpoint_radius),
                          std::move(wp));
  case TaskPointFactoryType::CUSTOM_KEYHOLE:
    return CreateASTPoint(KeyholeZone::CreateCustomKeyholeZone(location,
                                                               turnpoint_radius,
                                                               Angle::QuarterCircle()),
                          std::move(wp));
  case TaskPointFactoryType::DAEC_KEYHOLE:
    return CreateASTPoint(KeyholeZone::CreateDAeCKeyholeZone(location),
                          std::move(wp));
  case TaskPointFactoryType::BGAFIXEDCOURSE_SECTOR:
    return CreateASTPoint(KeyholeZone::CreateBGAFixedCourseZone(location),
                          std::move(wp));
  case TaskPointFactoryType::BGAENHANCEDOPTION_SECTOR:
    return CreateASTPoint(KeyholeZone::CreateBGAEnhancedOptionZone(location),
                          std::move(wp));
  case TaskPointFactoryType::AST_CYLINDER:
    return CreateASTPoint(std::make_unique<CylinderZone>(location,
                                                         turnpoint_radius),
                          std::move(wp));
  case TaskPointFactoryType::MAT_CYLINDER:
    return CreateAATPoint(CylinderZone::CreateMatCylinderZone(location),
                          std::move(wp));
  case TaskPointFactoryType::AAT_CYLINDER:
    return CreateAATPoint(std::make_unique<CylinderZone>(location,
                                                         turnpoint_radius),
                          std::move(wp));
  case TaskPointFactoryType::AAT_SEGMENT:
    return CreateAATPoint(std::make_unique<SectorZone>(location,
                                                       turnpoint_radius),
                          std::move(wp));
  case TaskPointFactoryType::AAT_ANNULAR_SECTOR:
    return CreateAATPoint(std::make_unique<AnnularSectorZone>(location,
                                                              turnpoint_radius),
                          std::move(wp));
  case TaskPointFactoryType::AAT_KEYHOLE:
    return CreateAATPoint(KeyholeZone::CreateCustomKeyholeZone(location,
                                                               turnpoint_radius,
                                                               Angle::QuarterCircle()),
                          std::move(wp));
  case TaskPointFactoryType::FINISH_SECTOR:
    return CreateFinish(SymmetricSectorZone::CreateFAISectorZone(location,
                                                                 false),
                        std::move(wp));
  case TaskPointFactoryType::FINISH_LINE:
    return CreateFinish(std::make_unique<LineSectorZone>(location,
                                                         finish_radius),
                        std::move(wp));
  case TaskPointFactoryType::FINISH_CYLINDER:
    return CreateFinish(std::make_unique<CylinderZone>(location,
                                                       finish_radius),
                        std::move(wp));

  case TaskPointFactoryType::COUNT:
    gcc_unreachable();
  }

  gcc_unreachable();
}

std::unique_ptr<StartPoint>
AbstractTaskFactory::CreateStart(const TaskPointFactoryType type,
                                 WaypointPtr wp) const noexcept
{
  assert(wp);

  if (!IsValidStartType(type))
    // error, invalid type!
    return NULL;

  return std::unique_ptr<StartPoint>((StartPoint *)CreatePoint(type, std::move(wp)).release());
}

std::unique_ptr<IntermediateTaskPoint>
AbstractTaskFactory::CreateIntermediate(const TaskPointFactoryType type,
                                        WaypointPtr wp) const noexcept
{
  assert(wp);

  if (!IsValidIntermediateType(type))
    return NULL;

  return std::unique_ptr<IntermediateTaskPoint>((IntermediateTaskPoint *)CreatePoint(type, std::move(wp)).release());
}

std::unique_ptr<FinishPoint>
AbstractTaskFactory::CreateFinish(const TaskPointFactoryType type,
                                  WaypointPtr wp) const noexcept
{
  assert(wp);

  if (!IsValidFinishType(type))
    return NULL;

  return std::unique_ptr<FinishPoint>((FinishPoint *)CreatePoint(type, std::move(wp)).release());
}

bool 
AbstractTaskFactory::Append(const OrderedTaskPoint &new_tp,
                            const bool auto_mutate) noexcept
{
  if (auto_mutate) {
    if (!task.TaskSize()) {
      // empty task, so add as a start point
      if (IsValidType(new_tp, task.TaskSize())) {
        // candidate is ok, so add it
        return task.Append(new_tp);
      } else {
        // candidate must be transformed into a startpoint
        return task.Append(*CreateStart(new_tp.GetWaypointPtr()));
      }
    }

    // non-empty task

    if (task.HasFinish()) {
      // old finish must be mutated into an intermediate point
      auto sp =
        CreateIntermediate(task.GetTaskPoint(task.TaskSize() - 1).GetWaypointPtr());

      task.Replace(*sp, task.TaskSize()-1);
    }

    if (IsValidType(new_tp, task.TaskSize()))
      // ok to append directly
      return task.Append(new_tp);

    // this point must be mutated into a finish
    return task.Append(*CreateFinish(new_tp.GetWaypointPtr()));
  }

  return task.Append(new_tp);
}

bool 
AbstractTaskFactory::Replace(const OrderedTaskPoint &new_tp,
                             const unsigned position,
                             const bool auto_mutate) noexcept
{
  if (auto_mutate) {
    if (IsValidType(new_tp, position))
      // ok to replace directly
      return task.Replace(new_tp, position);

    // will need to convert type of candidate
    std::unique_ptr<OrderedTaskPoint> tp;
    if (position == 0) {
      // candidate must be transformed into a startpoint
      tp = CreateStart(new_tp.GetWaypointPtr());
    } else if (IsPositionFinish(position) &&
               position + 1 == task.TaskSize()) {
      // this point must be mutated into a finish
      tp = CreateFinish(new_tp.GetWaypointPtr());
    } else {
      // this point must be mutated into an intermediate
      tp = CreateIntermediate(new_tp.GetWaypointPtr());
    }

    return task.Replace(*tp, position);
  }

  return task.Replace(new_tp, position);
}

bool 
AbstractTaskFactory::Insert(const OrderedTaskPoint &new_tp,
                            const unsigned position,
                            const bool auto_mutate) noexcept
{
  if (position >= task.TaskSize())
    return Append(new_tp, auto_mutate);

  if (auto_mutate) {
    if (position == 0) {
      if (task.HasStart()) {
        // old start must be mutated into an intermediate point
        auto sp =
          CreateIntermediate(task.GetTaskPoint(0).GetWaypointPtr());
        task.Replace(*sp, 0);
      }

      if (IsValidType(new_tp, 0)) {
        return task.Insert(new_tp, 0);
      } else {
        // candidate must be transformed into a startpoint
        return task.Insert(*CreateStart(new_tp.GetWaypointPtr()), 0);
      }
    } else {
      if (new_tp.IsIntermediatePoint()) {
        // candidate ok for direct insertion
        return task.Insert(new_tp, position);
      } else {
        // candidate must be transformed into a intermediatepoint
        return task.Insert(*CreateIntermediate(new_tp.GetWaypointPtr()),
                           position);
      }
    }
  }

  return task.Insert(new_tp, position);
}

bool
AbstractTaskFactory::Remove(const unsigned position,
                            const bool auto_mutate) noexcept
{
  if (position >= task.TaskSize())
    return false;

  if (auto_mutate) {
    if (position == 0) {
      // special case, remove start point..
      if (task.TaskSize() == 1) {
        return task.Remove(0);
      } else {
        // create new start point from next point
        auto sp = CreateStart(task.GetTaskPoint(1).GetWaypointPtr());
        return task.Remove(0) && task.Replace(*sp, 0);
      }
    } else if (IsPositionFinish(position - 1) &&
               position + 1 == task.TaskSize()) {
      // create new finish from previous point
      auto sp =
        CreateFinish(task.GetTaskPoint(position - 1).GetWaypointPtr());
      return task.Remove(position) &&
        task.Replace(*sp, position - 1);
    } else {
      // intermediate point deleted, nothing special to do
      return task.Remove(position);
    }
  }

  return task.Remove(position);
}

bool 
AbstractTaskFactory::Swap(const unsigned position,
                          const bool auto_mutate) noexcept
{
  if (task.TaskSize() <= 1)
    return false;
  if (position >= task.TaskSize() - 1)
    return false;

  const OrderedTaskPoint &orig = task.GetTaskPoint(position + 1);
  if (!Insert(orig, position, auto_mutate))
    return false;

  return Remove(position+2, auto_mutate);
}

const OrderedTaskPoint&
AbstractTaskFactory::Relocate(const unsigned position, 
                              WaypointPtr &&waypoint) noexcept
{
  task.Relocate(position, std::move(waypoint));
  return task.GetTaskPoint(position);
}

const OrderedTaskSettings &
AbstractTaskFactory::GetOrderedTaskSettings() const noexcept
{
  return task.GetOrderedTaskSettings();
}

void
AbstractTaskFactory::UpdateOrderedTaskSettings(OrderedTaskSettings &to) noexcept
{
  to.start_constraints.require_arm = constraints.start_requires_arm;
  to.finish_constraints.fai_finish = constraints.fai_finish;
}

bool 
AbstractTaskFactory::IsPositionIntermediate(const unsigned position) const noexcept
{
  if (IsPositionStart(position))
    return false;
  if (position >= constraints.max_points)
    return false;
  if (position + 1 < constraints.min_points)
    return true;

  if (constraints.IsFixedSize())
    return (position + 1 < constraints.max_points);
  else if (task.TaskSize() < constraints.min_points)
    return true;
  else
    return (position <= task.TaskSize());
}

bool 
AbstractTaskFactory::IsPositionFinish(const unsigned position) const noexcept
{
  if (IsPositionStart(position))
    return false;

  if (position + 1 < constraints.min_points)
    return false;
  if (position + 1 > constraints.max_points)
    return false;

  if (constraints.IsFixedSize())
    return (position + 1 == constraints.max_points);
  else
    return (position + 1 >= task.TaskSize());
}

bool
AbstractTaskFactory::ValidAbstractType(LegalAbstractPointType type, 
                                       const unsigned position) const noexcept
{
  const bool is_start = IsPositionStart(position);
  const bool is_finish = IsPositionFinish(position);
  const bool is_intermediate = IsPositionIntermediate(position);

  switch (type) {
  case POINT_START:
    return is_start;
  case POINT_FINISH:
    return is_finish;
  case POINT_AST:
    return is_intermediate &&
      (IsValidIntermediateType(TaskPointFactoryType::FAI_SECTOR) 
       || IsValidIntermediateType(TaskPointFactoryType::AST_CYLINDER)
       || IsValidIntermediateType(TaskPointFactoryType::CUSTOM_KEYHOLE)
       || IsValidIntermediateType(TaskPointFactoryType::DAEC_KEYHOLE)
       || IsValidIntermediateType(TaskPointFactoryType::BGAFIXEDCOURSE_SECTOR)
       || IsValidIntermediateType(TaskPointFactoryType::BGAENHANCEDOPTION_SECTOR));
  case POINT_AAT:
    return is_intermediate &&
      (IsValidIntermediateType(TaskPointFactoryType::AAT_CYLINDER)
       || IsValidIntermediateType(TaskPointFactoryType::MAT_CYLINDER)
       || IsValidIntermediateType(TaskPointFactoryType::AAT_SEGMENT)
       || IsValidIntermediateType(TaskPointFactoryType::AAT_ANNULAR_SECTOR));
  };
  return false;
}

bool 
AbstractTaskFactory::IsValidType(const OrderedTaskPoint &new_tp,
                               unsigned position) const noexcept
{
  switch (new_tp.GetType()) {
  case TaskPointType::START:
    return ValidAbstractType(POINT_START, position) &&
        IsValidStartType(GetType(new_tp));

  case TaskPointType::AST:
    return ValidAbstractType(POINT_AST, position) &&
        IsValidIntermediateType(GetType(new_tp));

  case TaskPointType::AAT:
    return ValidAbstractType(POINT_AAT, position)&&
        IsValidIntermediateType(GetType(new_tp));

  case TaskPointType::FINISH:
    return ValidAbstractType(POINT_FINISH, position)&&
        IsValidFinishType(GetType(new_tp));

  case TaskPointType::UNORDERED:
    /* obviously, when we check the type of an OrderedTaskPoint, we
       should never get type==UNORDERED */
    gcc_unreachable();
  }

  gcc_unreachable();
}

TaskPointFactoryType
AbstractTaskFactory::GetDefaultStartType() const noexcept
{
  TaskPointFactoryType type = behaviour.sector_defaults.start_type;
  if (!IsValidStartType(type) && !start_types.IsEmpty())
    type = start_types.UncheckedFirst();

  return type;
}

TaskPointFactoryType
AbstractTaskFactory::GetDefaultIntermediateType() const noexcept
{
  TaskPointFactoryType type = behaviour.sector_defaults.turnpoint_type;
  if (!IsValidIntermediateType(type) && !intermediate_types.IsEmpty())
    type = intermediate_types.UncheckedFirst();

  return type;
}

TaskPointFactoryType
AbstractTaskFactory::GetDefaultFinishType() const noexcept
{
  TaskPointFactoryType type = behaviour.sector_defaults.finish_type;
  if (!IsValidFinishType(type) && !finish_types.IsEmpty())
    type = finish_types.UncheckedFirst();

  return type;
}

LegalPointSet
AbstractTaskFactory::GetValidTypes(unsigned position) const noexcept
{
  LegalPointSet v;
  if (ValidAbstractType(POINT_START, position))
    v |= start_types;

  v |= GetValidIntermediateTypes(position);

  if (ValidAbstractType(POINT_FINISH, position))
    v |= finish_types;

  return v;
}

bool
AbstractTaskFactory::CheckAddFinish() noexcept
{
 if (task.TaskSize() < 2)
   return false;

 if (task.HasFinish())
   return false;

 auto fp = CreateFinish(task.GetPoint(task.TaskSize() - 1).GetWaypointPtr());
 assert(fp);
 Remove(task.TaskSize() - 1, false);
 Append(*fp, false);

 return true;
}

TaskValidationErrorSet
AbstractTaskFactory::ValidateFAIOZs() const noexcept
{
  bool valid = true;

  for (unsigned i = 0; i < task.TaskSize() && valid; i++) {
    const auto &tp = task.GetPoint(i);
    const auto ozsize = GetOZSize(tp.GetObservationZone());

    switch (GetType(tp)) {
    case TaskPointFactoryType::START_BGA:
    case TaskPointFactoryType::START_CYLINDER:
      valid = false;
      break;

    case TaskPointFactoryType::START_SECTOR:
      if (ozsize > 1000.01)
        valid = false;

      break;
    case TaskPointFactoryType::START_LINE:
      if (ozsize > 2000.01)
        valid = false;

      break;

    case TaskPointFactoryType::FAI_SECTOR:
      break;

    case TaskPointFactoryType::AST_CYLINDER:
      if (ozsize > 500.01)
        valid = false;

      break;

    case TaskPointFactoryType::CUSTOM_KEYHOLE:
    case TaskPointFactoryType::DAEC_KEYHOLE:
    case TaskPointFactoryType::BGAFIXEDCOURSE_SECTOR:
    case TaskPointFactoryType::BGAENHANCEDOPTION_SECTOR:
    case TaskPointFactoryType::MAT_CYLINDER:
    case TaskPointFactoryType::AAT_CYLINDER:
    case TaskPointFactoryType::AAT_SEGMENT:
    case TaskPointFactoryType::AAT_ANNULAR_SECTOR:
    case TaskPointFactoryType::AAT_KEYHOLE:
    case TaskPointFactoryType::SYMMETRIC_QUADRANT:
      valid = false;
      break;

    case TaskPointFactoryType::FINISH_SECTOR:
      break;
    case TaskPointFactoryType::FINISH_LINE:
      if (ozsize > 2000.01)
        valid = false;

      break;

    case TaskPointFactoryType::FINISH_CYLINDER:
      valid = false;
      break;

    case TaskPointFactoryType::COUNT:
      gcc_unreachable();
    }
  }

  TaskValidationErrorSet errors;

  if (!valid)
    errors |= TaskValidationErrorType::NON_FAI_OZS;

  return errors;
}

TaskValidationErrorSet
AbstractTaskFactory::ValidateMATOZs() const noexcept
{
  bool valid = true;

  for (unsigned i = 0; i < task.TaskSize() && valid; i++) {
    const OrderedTaskPoint &tp = task.GetPoint(i);

    switch (GetType(tp)) {
    case TaskPointFactoryType::START_CYLINDER:
    case TaskPointFactoryType::START_LINE:
    case TaskPointFactoryType::START_SECTOR:
      break;

    case TaskPointFactoryType::START_BGA:
      valid = false;
      break;

    case TaskPointFactoryType::MAT_CYLINDER:
      break;

    case TaskPointFactoryType::AAT_CYLINDER:
    case TaskPointFactoryType::FAI_SECTOR:
    case TaskPointFactoryType::AST_CYLINDER:
    case TaskPointFactoryType::CUSTOM_KEYHOLE:
    case TaskPointFactoryType::DAEC_KEYHOLE:
    case TaskPointFactoryType::BGAFIXEDCOURSE_SECTOR:
    case TaskPointFactoryType::BGAENHANCEDOPTION_SECTOR:
    case TaskPointFactoryType::AAT_SEGMENT:
    case TaskPointFactoryType::AAT_ANNULAR_SECTOR:
    case TaskPointFactoryType::AAT_KEYHOLE:
    case TaskPointFactoryType::FINISH_SECTOR:
    case TaskPointFactoryType::SYMMETRIC_QUADRANT:
      valid = false;
      break;

    case TaskPointFactoryType::FINISH_LINE:
    case TaskPointFactoryType::FINISH_CYLINDER:
      break;

    case TaskPointFactoryType::COUNT:
      gcc_unreachable();
    }
  }

  TaskValidationErrorSet errors;
  if (!valid)
    errors |= TaskValidationErrorType::NON_MAT_OZS;

  return errors;
}

TaskValidationErrorSet
AbstractTaskFactory::Validate() const noexcept
{
  TaskValidationErrorSet errors;

  if (task.TaskSize() == 0)
    errors |= TaskValidationErrorType::EMPTY_TASK;

  if (!task.HasStart())
    errors |= TaskValidationErrorType::NO_VALID_START;

  if (!task.HasFinish())
    errors |= TaskValidationErrorType::NO_VALID_FINISH;

  if (constraints.is_closed && !IsClosed())
    errors |= TaskValidationErrorType::TASK_NOT_CLOSED;

  if (constraints.IsFixedSize()) {
    if (task.TaskSize() != constraints.max_points)
      errors |= TaskValidationErrorType::INCORRECT_NUMBER_TURNPOINTS;
  } else {
    if (task.TaskSize() < constraints.min_points)
      errors |= TaskValidationErrorType::UNDER_MIN_TURNPOINTS;

    if (task.TaskSize() > constraints.max_points)
      errors |= TaskValidationErrorType::EXCEEDS_MAX_TURNPOINTS;
  }

  if (constraints.homogeneous_tps && !IsHomogeneous())
    errors |= TaskValidationErrorType::TASK_NOT_HOMOGENEOUS;

  return errors;
}

LegalPointSet
AbstractTaskFactory::GetValidIntermediateTypes(unsigned position) const noexcept
{
  if (!IsPositionIntermediate(position))
    return LegalPointSet();

  if (constraints.homogeneous_tps &&
      position > 1 && task.TaskSize() > 1) {
    TaskPointFactoryType type = GetType(task.GetPoint(1));
    if (IsValidIntermediateType(type))
      return LegalPointSet(type);
  }

  if (ValidAbstractType(POINT_AAT, position) ||
      ValidAbstractType(POINT_AST, position))
    return intermediate_types;

  return LegalPointSet();
}

bool 
AbstractTaskFactory::IsClosed() const noexcept
{
  if (task.TaskSize() < 3)
    return false;

  const auto wp_start = task.GetPoint(0).GetWaypointPtr();
  const auto wp_finish = task.GetPoint(task.TaskSize() - 1).GetWaypointPtr();

  return wp_start->location == wp_finish->location;
}

bool 
AbstractTaskFactory::IsUnique() const noexcept
{
  const unsigned size = task.TaskSize();
  for (unsigned i = 0; i + 1 < size; i++) {
    const auto wp_0 = task.GetPoint(i).GetWaypointPtr();

    for (unsigned j = i + 1; j < size; j++) {
      if (i == 0 && j + 1 == size) {
        // start point can be similar to finish point
      } else {
        const auto wp_1 = task.GetPoint(j).GetWaypointPtr();
        if (wp_1 == wp_0)
          return false;
      }
    }
  }
  return true;
}

bool
AbstractTaskFactory::IsHomogeneous() const noexcept
{
  bool valid = true;

  const unsigned size = task.TaskSize();

  if (size > 2) {
    TaskPointFactoryType homogtype = GetType(task.GetPoint(1));

    for (unsigned i = 2; i < size; i++) {
      const OrderedTaskPoint &tp = task.GetPoint(i);
      if (tp.GetType() == TaskPointType::FINISH) {
        ; // don't check a valid finish point
      } else {
        if (GetType(tp) != homogtype) {
          valid = false;
          break;
        }
      }
    }
  }

  return valid;
}

bool
AbstractTaskFactory::RemoveExcessTPsPerTaskType() noexcept
{
  bool changed = false;
  unsigned maxtp = constraints.max_points;
  while (maxtp < task.TaskSize()) {
    Remove(maxtp, false);
    changed = true;
  }
  return changed;
}

bool
AbstractTaskFactory::MutateTPsToTaskType() noexcept
{
  bool changed = RemoveExcessTPsPerTaskType();

  for (unsigned int i = 0; i < task.TaskSize(); i++) {
    const OrderedTaskPoint &tp = task.GetPoint(i);
    if (!IsValidType(tp, i) ||
        task.GetFactoryType() == TaskFactoryType::MAT ||
        task.GetFactoryType() == TaskFactoryType::FAI_GENERAL) {

      TaskPointFactoryType newtype = GetMutatedPointType(tp);
      if (IsPositionFinish(i)) {

        if (!IsValidFinishType(newtype))
          newtype = GetDefaultFinishType();

        auto fp = CreateMutatedPoint(tp, newtype);
        assert(fp);
        if (Replace(*fp, i, true))
          changed = true;
      } else if (i == 0) {
        if (!IsValidStartType(newtype))
          newtype = GetDefaultStartType();

        auto sp = CreateMutatedPoint(tp, newtype);
        assert(sp);
        if (Replace(*sp, i, true))
          changed = true;
      } else {

        if (!IsValidIntermediateType(newtype))
          newtype = GetDefaultIntermediateType();

        auto tpnew = CreateMutatedPoint(tp, newtype);
        if (Replace(*tpnew, i, true))
          changed = true;
      }
    }
  }

  changed |= MutateClosedFinishPerTaskType();
  return changed;
}

bool
AbstractTaskFactory::MutateClosedFinishPerTaskType() noexcept
{
  if (task.TaskSize() < 2)
    return false;

  if (!IsPositionFinish(task.TaskSize() - 1))
    return false;

  bool changed = false;

  if (constraints.is_closed) {
    if (!IsClosed()) {
      const OrderedTaskPoint &tp = task.GetPoint(task.TaskSize() - 1);
      if (tp.GetType() == TaskPointType::FINISH) {
        auto fp = CreateFinish(task.GetPoint(0).GetWaypointPtr());
        assert(fp);
        Remove(task.TaskSize() - 1, false);
        Append(*fp, false);
        changed = true;
      }
    }
  }
  return changed;
}

bool 
AbstractTaskFactory::AppendOptionalStart(WaypointPtr wp) noexcept
{
  std::unique_ptr<OrderedTaskPoint> tp;
  if (task.TaskSize())
    tp = task.GetPoint(0).Clone(behaviour, GetOrderedTaskSettings(),
                                std::move(wp));
  else
    tp = CreateStart(std::move(wp));

  if (!tp)
    return false; // should never happen

  return task.AppendOptionalStart(*tp);
}

bool
AbstractTaskFactory::AppendOptionalStart(const OrderedTaskPoint &new_tp,
                                         const bool auto_mutate) noexcept
{
  if (auto_mutate && !IsValidType(new_tp, 0)) {
    // candidate must be transformed into a startpoint of appropriate type
    auto sp = CreateStart(new_tp.GetWaypointPtr());
    return task.AppendOptionalStart(*sp);
  }
  // ok to add directly
  return task.AppendOptionalStart(new_tp);
}

void
AbstractTaskFactory::UpdateStatsGeometry() noexcept
{
  task.UpdateStatsGeometry();
}

void
AbstractTaskFactory::UpdateGeometry() noexcept
{
  task.UpdateGeometry();
}
