// SPDX-License-Identifier: GPL-2.0-or-later
// Copyright The XCSoar Project

#include "InputEvents.hpp"
#include "InputConfig.hpp"
#include "ui/event/KeyCode.hpp"
#include "Asset.hpp"
#include "util/Macros.hpp"

#include <cassert>
#include <cstdint>

#ifdef _WIN32
/* we don't need that WIN32 API function, it's a macro that will
   disallow using InputConfig::AppendMenu() */
#undef AppendMenu
#endif

/**
 * For data generated by xci2cpp.pl.
 */
struct flat_event_map {
  unsigned char mode;

#ifdef ENABLE_SDL
#if defined(SDLK_SCANCODE_MASK) && SDLK_SCANCODE_MASK >= 0x10000
  /* need a bigger type for SDL 1.3+ */
  unsigned key;
#else
  unsigned short key;
#endif
#elif defined(USE_POLL_EVENT)
  uint16_t key;
#else
  unsigned char key;
#endif

  unsigned short event;
};

/**
 * For data generated by xci2cpp.pl.
 */
struct flat_label {
  unsigned char mode, location;
  unsigned short event;
  const TCHAR *label;
};

struct flat_gesture_map {
  unsigned char mode;
  unsigned short event;
  const TCHAR *data;
};

// Make a new label (add to the end each time)
// NOTE: String must already be copied (allows us to use literals
// without taking up more data - but when loading from file must copy string
static void
makeLabel(InputConfig &input_config,
          InputEvents::Mode mode_id, const TCHAR* label,
          unsigned location, unsigned event_id)
{
  input_config.AppendMenu(mode_id, label, location, event_id);
}

static void
apply_defaults(InputConfig &input_config,
               const TCHAR *const* default_modes,
               const InputConfig::Event *default_events,
               unsigned num_default_events,
               const flat_gesture_map *default_gesture2event,
               const flat_event_map *default_key2event,
               const flat_event_map *default_gc2event,
               const flat_event_map *default_n2event,
               const flat_label *default_labels)
{
  assert(num_default_events <= InputConfig::MAX_EVENTS);

  input_config.SetDefaults();
  while (*default_modes != NULL)
    input_config.AppendMode(*default_modes++);

  input_config.events.resize(num_default_events + 1);
  std::copy_n(default_events, num_default_events,
              input_config.events.begin() + 1);

  while (default_gesture2event->event > 0) {
    input_config.Gesture2Event.Add(default_gesture2event->data,
                                   default_gesture2event->event);
    ++default_gesture2event;
  }
  
  while (default_key2event->event > 0) {
    input_config.SetKeyEvent(default_key2event->mode,
                             default_key2event->key,
                             default_key2event->event);
    ++default_key2event;
  }

  while (default_gc2event->event > 0) {
    input_config.GC2Event[default_gc2event->key] =
      default_gc2event->event;
    ++default_gc2event;
  }

  while (default_n2event->event > 0) {
    input_config.N2Event[default_n2event->key] =
      default_n2event->event;
    ++default_n2event;
  }

  while (default_labels->label != NULL) {
    makeLabel(input_config,
              (InputEvents::Mode)default_labels->mode,
              default_labels->label,
              default_labels->location, default_labels->event);
    ++default_labels;
  }
}

// Read the data files
void
InputEvents::LoadDefaults(InputConfig &input_config)
{
  // Get defaults
  input_config.SetDefaults();

#include "InputEvents_default.cpp"
  apply_defaults(input_config,
                 default_modes,
                 default_events,
                 ARRAY_SIZE(default_events),
                 default_gesture2event,
                 default_key2event, default_gc2event, default_n2event,
                 default_labels);
}
